' NIM for CMM2
' Rev 1.0.0 William M Leue 10/24/2020
' Can be played with 2 players or 1 vs the computer
' Can be played normal or misere.

option default integer
option base 1

' Basic Game Constants
const MAX_HEAPS = 10
const MAX_BEADS = 9
const NORMAL = 1
const MISERE = 2
const HUMAN = 1
const COMPUTER = 2
const PLAYER1 = 1
const PLAYER2 = 2

' Graphic Constants
const BRAD = 5
const BSEP = 20
const HBSIZE = 100
const HMARGIN = 40
const VMARGIN = 40
const NHPERROW = 4
const CBOXX = 20
const CBOXY = 450
const CBOXW = 700
const CBOXH = 100
const CBOXNX = 500
const CBOXNY = 470
const CBOXNW = 100
const CBOXNH = 60
const BGCOLOR = RGB(120, 120, 120)
const HCOLOR = RGB(40, 40, 40)
const HCOLOR2 = RGB(160,  160, 160)
const HCOLOR3 = RGB(20, 20, 20)

' Keyboard Constants
const NUM_KEYS = 7
const UP = 1
const RIGHT = 2
const DOWN = 3
const LEFT = 4
const PLUS = 5
const MINUS = 6
const ENTER = 7

' Control Box Constants
const CB_P1MOVEA = 1
const CB_P1MOVEB = 2
const CB_CMOVEA  = 3
const CB_CMOVEB  = 4
const CB_P1GWON  = 5
const CB_P2MOVEA = 6
const CB_P2MOVEB = 7
const CB_P2MOVEC = 8
const CB_P2GWON  = 9

' Globals
dim num_heaps = 0
dim start_beads = 0
dim num_players = 1
dim pmode = NORMAL
dim random = 0
dim heaps(MAX_HEAPS)
dim bead_colors(MAX_HEAPS)
dim bead_hilites(MAX_HEAPS)
dim bead_lolites(MAX_HEAPS)
dim game_running = 0
dim first_player = COMPUTER
dim last_player = 0
dim keys(NUM_KEYS) = (128, 131, 129, 130, 43, 45, 13)
dim selected_heap = 0
dim prev_selected_heap = 0
dim selected_beads = 0
dim heap_locs(MAX_HEAPS, 2)
dim bead_locs(MAX_HEAPS, MAX_BEADS, 2)
dim current_player = 0

' Main Program
'open "debug.txt" for output as #1
SetGraphics
do
  GetUserParameters
  NewGame
  if num_players = 1 then
    OnePlayerGameLoop
  else
    current_player = 1
    TwoPlayerGameLoop
  end if
  NextGameLoop
loop
'close #1
end

' Set up graphics mode
sub SetGraphics
  mode 1,8
end sub

' Get the number of players and game
' mode from the user(s).
sub GetUserParameters
  local ok, i
  local ans$

  cls
  text MM.HRES\2, 10, "NIM!", "CT", 5
  text 0, 60, ""
  ok = 1
  print "Need Instructions for Playing NIM?: ";
  input ans$
  if LEFT$(UCASE$(ans$), 1) = "Y" then
    ShowInstructions
  end if
  do
    ok = 1
    print "How many players? (1 or 2): ";
    input ans$
    num_players = val(ans$)
    if num_players = 0 then end
    if num_players > 2 then ok = 0
  loop until ok = 1
  do
    ok = 1
    print "Normal (last player wins) or Misere (last player loses) game? (1 or 2): ";
    input ans$
    pmode = val(ans$)
    if pmode = 0 then end
    if pmode > MISERE then ok = 0
  loop until ok = 1
  do
    ok = 1
    print "Do you want random heaps and beads (1) or choose your own (2): ";
    input ans$
    random = val(ans$)
    if random = 0 then end
    if random > 2 then ok = 0
  loop until ok = 1
  if random = 2 then
    do
      ok = 1
      print "How many heaps? (1-" + str$(MAX_HEAPS) + "): ";
      input ans$
      num_heaps = val(ans$)
      if num_heaps = 0 then end
      if num_heaps < 1 or num_heaps > MAX_HEAPS then ok = 0
    loop until ok = 1
    do
      ok = 1
      print "How many beads per heap? (1-" + str$(MAX_BEADS) + "): ";
      input ans$
      start_beads = val(ans$)
      if start_beads = 0 then end
      if start_beads < 1 or start_beads > MAX_BEADS then ok = 0
    loop until ok = 1
  end if
end sub

' Make the Bead Colors
sub MakeColors
  local i, r, g, b
  local float hue, hinc
    hinc = 360.0/num_heaps 
    hue = 0.0
    for i = 1 to num_heaps
      bead_colors(i) = getRGBColor(hue, 0.5, 0.7)
      bead_hilites(i) = getRGBColor(hue, 0.3, 1.0)
      bead_lolites(i) = getRGBColor(hue, 0.7, 0.2)
      hue = hue+hinc
    next i
end sub

' set up a new game
sub NewGame
  local i
  if random = 2 then
    for i = 1 to num_heaps
      heaps(i) = start_beads
    next i
  else
    num_heaps = 5 + int(rnd()*6)
    for i = 1 to num_heaps
      heaps(i) = 1 + int(rnd()*9)
    next i
  end if
  MakeHeapLocations
  for i = 1 to num_heaps
    MakeBeadLocations i, heaps(i)
  next i
  if first_player = HUMAN then
    first_player = COMPUTER
  else
    first_player = HUMAN
  end if
  MakeColors
  DrawGame
  game_running = 1
end sub

' Make all the Heap locations
sub MakeHeapLocations
  local i, h, n, x, y, nrows, row, col

  nrows = (num_heaps+NHPERROW-1)\NHPERROW
  h = 0
  for row = 1 to nrows
    y = VMARGIN + (row-1)*(VMARGIN+HBSIZE)
    n = NHPERROW
    if row = nrows then n = num_heaps - (nrows-1)*NHPERROW
    for col = 1 to n
      x = HMARGIN + (col-1)*(HMARGIN+HBSIZE)  
      h = h+1
      heap_locs(h,1) = x : heap_locs(h,2) = y
    next col
  next row
end sub
  
' Draw all the game components:
' The heaps and the control boxes.
sub DrawGame
  local i, x, y

  cls
  box 0,0, MM.HRES, MM.VRES,,, BGCOLOR
  for i = 1 to num_heaps
    x = heap_locs(i, 1) : y = heap_locs(i, 2)
    DrawHeap i, x, y 
  next i
end sub

' Draw a Heap at the specified location with the current
' number of beads.
sub DrawHeap h, x, y
  local bx, by, n, cx, cy, b
  n = heaps(h)
  cx = x + HBSIZE\2
  cy = y + HBSIZE\2
  circle cx, cy-5, HBSIZE\2,,, HCOLOR3, HCOLOR3
  circle cx, cy+5, HBSIZE\2,,, HCOLOR2, HCOLOR2
  circle cx, cy, HBSIZE\2,,, HCOLOR, HCOLOR
  for b = 1 to n
    bx = bead_locs(h, b, 1) : by = bead_locs(h, b, 2)
    circle bx, by+2, BRAD,,, bead_hilites(h), bead_hilites(h)
    circle bx, by-2, BRAD,,, bead_lolites(h), bead_lolites(h)    
    circle bx, by, BRAD,,, bead_colors(h), bead_colors(h)
  next b
  text x+(BRAD*BSEP)\2, y-25, str$(h)
end sub

' Draw a dialog box for player interaction and information
' during a game.
sub DrawControlBox which, num, pnum
  local theHeap, theBeads
  local msg$

  box CBOXX, CBOXY, CBOXW, CBOXH,, RGB(BLACK), RGB(BLACK)
  if which = 0 then exit sub

  box CBOXX, CBOXY, CBOXW, CBOXH
  box CBOXNX, CBOXNY, CBOXNW, CBOXNH
    
  select case which
  case CB_P1MOVEA
    text CBOXX+10, CBOXY+20, "Use the Arrow Keys to Choose a Heap"
    text CBOXX+10, CBOXY+40, "Press the Enter Key when you have chosen."
    text CBOXNX+20, CBOXNY+10, "   ", "LT", 5
    text CBOXNX+20, CBOXNY+10, str$(num), "LT", 5
  case CB_P1MOVEB
    text CBOXX+10, CBOXY+20, "Use the Arrow Keys to Choose the"
    text CBOXX+10, CBOXY+40, "number of beads to remove from the chosen heap."
    text CBOXX+10, CBOXY+60, "Press the Enter Key when you have chosen."
    text CBOXNX+20, CBOXNY+10, "   ", "LT", 5
    text CBOXNX+20, CBOXNY+10, str$(num), "LT", 5
  case CB_CMOVEA
    text CBOXX+10, CBOXY+20, "Computer is deciding its move, please wait..."
  case CB_CMOVEB
    theHeap = num\100
    theBeads = (num - theHeap*100)
    msg$ = "Computer removes " + str$(theBeads) + " beads from Heap " + str$(theHeap)
    text CBOXX+10, CBOXY+20, msg$
  case CB_P1GWON
    if num = HUMAN then
      msg$ = "Game Over -- You Win"
    else
      msg$ = "Game Over -- Computer Wins"
    end if      
    text CBOXX+10, CBOXY+20, msg$
    text CBOXX+10, CBOXY+40, "Press the '+' key for a new game"
    text CBOXX+10, CBOXY+60, "Press the '-' key to quit"
  case CB_P2MOVEA
    if pnum = 1 then
      msg$ = "Player 1's Move"
    else
      msg$ = "Player 2's Move"
    end if
    text CBOXX+10, CBOXY+20, msg$
    text CBOXX+10, CBOXY+40, "Use the Arrow Keys to Choose a Heap"
    text CBOXX+10, CBOXY+60, "Press the Enter Key when you have chosen."
    text CBOXNX+20, CBOXNY+10, "   ", "LT", 5
    text CBOXNX+20, CBOXNY+10, str$(num), "LT", 5
  case CB_P2MOVEB
    if pnum = 1 then
      msg$ = "Player 1's Move"
    else
      msg$ = "Player 2's Move"
    end if
    text CBOXX+10, CBOXY+20, msg$
    text CBOXX+10, CBOXY+40, "Use the Arrow Keys to Choose the"
    text CBOXX+10, CBOXY+60, "number of beads to remove from the chosen heap."
    text CBOXX+10, CBOXY+80, "Press the Enter Key when you have chosen."
    text CBOXNX+20, CBOXNY+10, "   ", "LT", 5
    text CBOXNX+20, CBOXNY+10, str$(num), "LT", 5
  case CB_P2MOVEC
    theHeap = num\100
    theBeads = (num - theHeap*100)
    if pnum = 1 then
      msg$ = "Player 1"
    else
      msg$ = "Player 2 "
    end if
    msg$ = msg$ + " removed " + str$(theBeads) + " beads from heap " + str$(theHeap)
    text CBOXX+10, CBOXY+20, msg$
  case CB_P2GWON
    if num = 1 then
      msg$ = "Player 1 Wins!"
    else
      msg$ = "Player 2 Wins!"
    end if
    text CBOXX+10, CBOXY+20, msg$
    text CBOXX+10, CBOXY+40, "Press the '+' key for a new game"
    text CBOXX+10, CBOXY+60, "Press the '-' key to quit"
  end select  
 
end sub

' Hilite the selected heap with a halo
sub HiliteHeap which
  local px, py, hsize, hoff
  local x, y
  hsize = HBSIZE+20
  hoff = 10
  if prev_selected_heap > 0 then
    px = heap_locs(prev_selected_heap, 1) - hoff
    py = heap_locs(prev_selected_heap, 2) - hoff
    box px, py, hsize, hsize,, BGCOLOR
  end if
  prev_selected_heap = which
  if which > 0 then
    x = heap_locs(which, 1) - hoff : y = heap_locs(which, 2) - hoff
    box x, y, hsize, hsize,, RGB(YELLOW)
  end if
end sub

' Find a randomized location for beads in a heap circle
' (Only called once per game)
sub MakeBeadLocations which, num
  local i, j, cxm cy, bx, by, ok, xp, m
  local float r, a
  cx = heap_locs(which, 1) + HBSIZE\2
  cy = heap_locs(which, 2) + HBSIZE\2
  m = 8
  for i = 1 to num
    do
      ok = 1
      r = 0.42*rnd()*HBSIZE
      a = RAD(rnd()*360.0)
      bx = cx + int(r*cos(a))
      by = cy - int(r*sin(a))
      if i > 1 then
        for j = 1 to i-1
          xp = 0
          if bx >= bead_locs(which, j, 1)-(BRAD+m) and bx <= bead_locs(which, j, 1)+(BRAD+m) then
            xp = 1
          end if
          if by >= bead_locs(which, j, 2)-(BRAD+m) and by <= bead_locs(which, j, 2)+(BRAD+m) then
            if xp = 1 then ok = 0
          end if
          if ok = 0 then exit for
        next j
      end if
    loop until ok = 1
    bead_locs(which, i, 1) = bx : bead_locs(which, i, 2) = by
  next i
end sub

' Process User inputs and play the game: user vs computer
sub OnePlayerGameLoop
  local i, key, cmd, state, nbeads
  selected_heap = 1
  DrawControlBox 1, selected_heap
  HiliteHeap selected_heap
  state = 1
  do while game_running
    z$ = INKEY$
    do
      z$ = INKEY$
    loop until z$ <> ""
    key = asc(z$)
    cmd = -1
    for i = 1 to NUM_KEYS
      if key = keys(i) then
        cmd = i
        exit for
      end if
    next i
    select case cmd
      case UP to RIGHT
        if state  = 1 then
          selected_heap = selected_heap+1
          if selected_heap > num_heaps then selected_heap = 1
          DrawControlBox CB_P1MOVEA, selected_heap
          HiliteHeap selected_heap
        else
          selected_beads = selected_beads+1
          if selected_beads > nbeads then selected_beads = nbeads
          DrawControlBox CB_P1MOVEB, selected_beads
        end if
      case DOWN to LEFT
        if state = 1 then
          selected_heap = selected_heap-1
          if selected_heap < 1 then selected_heap = num_heaps
          DrawControlBox CB_P1MOVEA, selected_heap
          HiliteHeap selected_heap
        else
          selected_beads = selected_beads-1
          if selected_beads < 1 then selected_beads = 1
          DrawControlBox CB_P1MOVEB, selected_beads
        end if
      case PLUS
        exit sub
      case MINUS
        'close #1
        end
      case ENTER
        if state = 1 then
          state = 2
          nbeads = heaps(selected_heap)
          selected_beads = 1
          DrawControlBox CB_P1MOVEA, 1
        else
          DrawControlBox 0, 0
          heaps(selected_heap) = heaps(selected_heap) - selected_beads
          selected_heap = 0
          prev_selected_heap = 0
          selected_beads = 0
          DrawGame
          HiliteHeap 0
          DrawControlBox CB_CMOVEA, 0
          last_player = HUMAN
          CheckWin
          if NOT game_running then continue do
          GetComputerMove selected_heap, selected_beads
          pause 3000
          if selected_heap > 0 then
            DrawGame
            DrawControlBox CB_CMOVEB, 100*selected_heap+selected_beads
            HiliteHeap selected_heap
            pause 3000
            last_player = COMPUTER
            CheckWin
            if not game_running then continue do
            state = 1
            for i = 1 to num_heaps
              if heaps(i)  > 0 then
                selected_heap = i
                exit for
              end if
            next i
            selected_beads = 1
            DrawGame
            DrawControlBox CB_P1MOVEA, selected_heap 
            HiliteHeap selected_heap
          else
            ERROR "Computer failed to find a move"
          end if  
        end if  
      case else
        ' ignore
    end select
  loop  
end sub

' Handle events for a 2-player game
sub TwoPlayerGameLoop
  local i, key, cmd, state, nbeads

  state = 1
  selected_heap = 1
  do while game_running
    if state = 1 then
      DrawControlBox CB_P2MOVEA, selected_heap, current_player
    else 
      DrawControlBox CB_P2MOVEB, selected_beads, current_player
    end if
    HiliteHeap selected_heap
    z$ = INKEY$
    do
      z$ = INKEY$
    loop until z$ <> ""
    key = asc(z$)
    cmd = -1
    for i = 1 to NUM_KEYS
      if key = keys(i) then
        cmd = i
        exit for
      end if
    next i
    select case cmd
      case UP to RIGHT
        if state  = 1 then
          selected_heap = selected_heap+1
          if selected_heap > num_heaps then selected_heap = 1
          DrawControlBox CB_P2MOVEA, selected_heap, current_player
          HiliteHeap selected_heap
        else
          selected_beads = selected_beads+1
          if selected_beads > nbeads then selected_beads = nbeads
          DrawControlBox CB_P2MOVEB, selected_beads, current_player
        end if
      case DOWN to LEFT
        if state = 1 then
          selected_heap = selected_heap-1
          if selected_heap < 1 then selected_heap = num_heaps
          DrawControlBox CB_P2MOVEA, selected_heap, current_player
          HiliteHeap selected_heap
        else
          selected_beads = selected_beads-1
          if selected_beads < 1 then selected_beads = 1
          DrawControlBox CB_P2MOVEB, selected_beads, current_player
        end if
      case PLUS
        ' ignore
      case MINUS
        'close #1
        end
      case ENTER
        if state = 1 then
          state = 2
          nbeads = heaps(selected_heap)
          selected_beads = 1
          DrawControlBox CB_P2MOVEB, selected_beads, current_player
        else
          heaps(selected_heap) = heaps(selected_heap) - selected_beads
          DrawGame
          DrawControlBox CB_P2MOVEC, 100*selected_heap+selected_beads, current_player
          HiliteHeap selected_heap
          pause 3000
          last_player = current_player
          CheckWin
          if NOT game_running then exit sub
          state = 1
          for i = 1 to num_heaps
            if heaps(i)  > 0 then
              selected_heap = i
              exit for
            end if
          next i
          selected_beads = 1
          DrawGame
          if current_player = 1 then
            current_player = 2
          else
            current_player = 1
          end if
          DrawControlBox CB_P2MOVEA, selected_heap, current_player
          HiliteHeap selected_heap
        end if
    end select  
  loop  
end sub

' See if another game is desired, then loop back to
' parameter screen or quit.
sub NextGameLoop
  local i, key, cmd
  local z$
  z$ = INKEY$
  do
    do
      z$ = INKEY$
    loop until z$ <> ""
    key = asc(z$)
    for i = 1 to NUM_KEYS
      if key = keys(i) then
        cmd = i
        exit for
      end if
    next i
    select case cmd
      case PLUS
        exit sub
      case MINUS
        'close #1
        end
      else
        ' ignore
    end select  
  loop 
end sub

' The NimSum is the exclusive OR of the
' binary values of all the heap sizes.
function NimSum()
  local i, b, s
  s = 0
  for i = 1 to num_heaps
    b = heaps(i)
    s = s xor b
  next i
  NimSum = s
end function

' The computer move is chosen so as to
' make the NimSum zero for Normal games,
' and one for Misere games. If a safe
' move is not available, a minimal move
' is made.
sub GetComputerMove heap, nbeads
  local i, j, s
  for i = 1 to num_heaps
    for j = 1 to heaps(i)
      heaps(i) = heaps(i) - j
      s = NimSum()
      if pmode = NORMAL then
        if s = 0 then
          heap = i
          nbeads = j
          exit sub
        end if
      else
        if s = 1 then
          heap = i
          nbeads = j
          exit sub
        end if
      end if
      heaps(i) = heaps(i) + j
    next j
  next i
  for i = 1 to num_heaps
    if heaps(i) > 1 then
      heap = i
      nbeads = 1
      heaps(i) = heaps(i) - 1
      exit sub
    end if
  next i
  for i = 1 to num_heaps
    if heaps(i) > 0 then
      heap = i
      nbeads = 1
      heaps(i) = heaps(i) - 1
      exit sub
    end if
  next i
end sub

' CheckWin: looks for a win.
' Halts game and announces winner
sub CheckWin
  local i, n
  n = 0
  for i = 1 to num_heaps
    n = n+heaps(i)
  next i
  if n = 0 then
    game_running = 0
    if num_players = 1 then
      if pmode = NORMAL then
        if last_player = HUMAN then
          DrawControlBox CB_P1GWON, HUMAN
        else
          DrawControlBox CB_P1GWON, COMPUTER
        end if
      else
        if last_player = HUMAN then
          DrawControlBox CB_P1GWON, COMPUTER
        else
          DrawControlBox CB_P1GWON, HUMAN
        end if
      end if    
    else
      if pmode = NORMAL then
        if last_player = PLAYER1 then
          DrawControlBox CB_P2GWON, PLAYER1
        else
          DrawControlBox CB_P2GWON, PLAYER2
        end if
      else
        if last_player = PLAYER1 then
          DrawControlBox CB_P2GWON, PLAYER2
        else
          DrawControlBox CB_P2GWON, PLAYER1
        end if
      end if
    end if
  end if
end sub

' Convert an HSV value to its RGB equivalent
' The S and V values must be in range 0..1; the H value must
' be in range 0..360. The RGB values will be in range 0..255.
sub HSV2RGB h, s, v, r, g, b
  local hh, i, f, p, q, t, x, c

  c = v*s
  hh = h/60.0
  i = int(hh)
  f = hh - i
  p = v*(1-s)
  q = v*(1-s*f)
  t = v*(1-s*(1-f))
  x = c*(1.0 - hi MOD 2 - 1)
  
  select case i
    case 0
      rp = v : gp = t : bp = p
    case 1
      rp = q : gp = v : bp = p
    case 2
      rp = p : gp = v : bp = t
    case 3
      rp = p : gp = q : bp = v
    case 4
      rp = t : gp = p : bp = v
    case 5
      rp = v : gp = p : bp = q
  end select
  r = rp*255 : g = gp*255 : b = bp*255
end sub

' function to return an RGB color, given the h, s, and v
' values as input parameters. The S and V values must be
' in the range 0..1; the H value must be in the range
' 0..360. The output value will be a 24-bit RGB color.
function GetRGBColor(h, s, v)
  local r, g, b, c

  HSV2RGB h, s, v, r, g, b
  c = RGB(r ,g, b)
  GetRGBColor = c
end function

' Print Game Instructions
sub ShowInstructions
  cls
  print "How to Play NIM"
  print ""
  print "A NIM game consists of several 'heaps' (think cups or pits). Each 'heap' contains"
  print " several beads. Each heap can contain a different number of beads."
  print ""
  print "On your turn, you can take any number of beads from a chosen heap. You must take at"
  print "least one bead, but you can take any number, including ALL the beads remaining in"
  print "that heap. However, you can only choose a single heap on each turn."
  print ""
  print "Each player takes turns choosing a heap and removing one or more beads. The game ends"
  print "when all the beads are gone. In a normal NIM game, the person who takes the last bead"
  print "WINS the game. If you are playing in 'misere' mode, then the person who takes the last"
  print "bead LOSES the game.
  print ""
  print "In this computer version of NIM, you can choose to play with the computer as your"
  print "opponent, or you can play with two human players taking turns."
  print ""
  print "You can choose to play a NORMAL game or a MISERE game."
  print ""
  print "You can also let the computer select a random number of heaps and beads in each heap."
  print "Or you can decide how many heaps and how many beads in each heap should be there."
  print "This is often helpful if you are just learning how to play NIM.
  print ""
  print ""
  print "Press any keyboard key to continue."
  z$ = INKEY$
  do
    z$ = INKEY$
  loop until z$ <> ""
end sub
